#!/usr/bin/env node

/**
 * Setup script to create upload directories with proper permissions
 * Run this script after deployment: node setup-upload-dirs.js
 */

const { mkdir, chmod } = require('fs/promises');
const { join } = require('path');
const { existsSync } = require('fs');

async function setupDirectories() {
  const directories = [
    'public',
    'public/logos',
    'data'
  ];

  console.log('Setting up upload directories...\n');

  for (const dir of directories) {
    const dirPath = join(process.cwd(), dir);
    
    try {
      if (!existsSync(dirPath)) {
        await mkdir(dirPath, { recursive: true, mode: 0o755 });
        console.log(`✓ Created directory: ${dir}`);
      } else {
        console.log(`✓ Directory exists: ${dir}`);
      }
      
      // Set permissions (755 = rwxr-xr-x)
      try {
        await chmod(dirPath, 0o755);
        console.log(`✓ Set permissions (755) for: ${dir}`);
      } catch (error) {
        console.log(`⚠ Could not set permissions for ${dir}: ${error.message}`);
      }
    } catch (error) {
      console.error(`✗ Error setting up ${dir}:`, error.message);
    }
  }

  console.log('\n✓ Setup complete!');
  console.log('\nDirectories ready for file uploads:');
  console.log('  - public/ (site logo, favicon)');
  console.log('  - public/logos/ (partner logos)');
  console.log('  - data/ (JSON data files)');
}

setupDirectories().catch((error) => {
  console.error('Setup failed:', error);
  process.exit(1);
});

