import type { Metadata } from "next";
import { headers } from "next/headers";
import { Suspense } from "react";
import type { Partner } from "@/types/data";
import { createMetadata, BASE_URL } from "@/lib/metadata";
import { shouldShowAllPartners, shouldShowOnlyUnlicensed } from "@/lib/detection";
import { PartnerShowcase } from "@/components/PartnerShowcase";
import { VisibilityDebug } from "@/components/VisibilityDebug";
import { getPartners, getSiteInfo } from "@/lib/data";

type SearchParamsRecord = Record<string, string | string[] | undefined>;

type HomePageProps = {
  searchParams?: SearchParamsRecord | Promise<SearchParamsRecord>;
};

export async function generateMetadata(): Promise<Metadata> {
  const siteInfo = getSiteInfo();
  return createMetadata({
    title: siteInfo.browserTitle,
    description:
      "Compare curated UK partners, explore verified welcome bonuses, and claim offers that meet quality standards.",
    path: "/",
  });
}

export default async function Home({ searchParams }: HomePageProps) {
  // Read data at runtime (not bundled)
  const partners = getPartners();
  const siteInfo = getSiteInfo();
  const licensedPartners = partners.filter((partner) => partner.licensed);
  const unlicensedPartners = partners.filter((partner) => !partner.licensed);
  
  const resolvedSearchParams = await Promise.resolve(searchParams ?? {});
  
  // Extract gclid for affiliate link tracking (use first available Google Ads param)
  // This is used ONLY for client-side affiliate link generation, never exposed in HTML
  const gclidValueRaw = resolvedSearchParams.gclid || 
                        resolvedSearchParams.gad_source || 
                        resolvedSearchParams.gad_campaignid ||
                        resolvedSearchParams.gbraid;
  const gclid = Array.isArray(gclidValueRaw)
    ? gclidValueRaw[0]
    : gclidValueRaw ?? null;

  const headersList = await headers();
  const userAgent = headersList.get("user-agent");
  const visitorTypeHeader = headersList.get("x-visitor-type");
  const referer = headersList.get("referer");
  
  // All detection logic runs server-side and is NEVER exposed in HTML output
  // These values are only used for internal logic, never rendered to the page
  
  // NEW RULE: If user has gclid (Google Ads), show ONLY unlicensed brands
  const showOnlyUnlicensed = shouldShowOnlyUnlicensed(resolvedSearchParams);
  
  const showAllPartners = shouldShowAllPartners({
    userAgent,
    visitorTypeHeader,
    searchParams: resolvedSearchParams,
    referer,
  });
  
  // For debugging only (development mode) - never exposed in production
  const isDevelopment = process.env.NODE_ENV === "development";

  const organizationLd = {
    "@context": "https://schema.org",
    "@type": "Organization",
    name: siteInfo.siteName,
    url: `${BASE_URL}/`,
    logo: `${BASE_URL}${siteInfo.siteLogo}`,
    email: siteInfo.email,
    address: {
      "@type": "PostalAddress",
      addressLocality: siteInfo.address,
      addressCountry: "UK",
    },
    contactPoint: [
      {
        "@type": "ContactPoint",
        telephone: siteInfo.phone,
        contactType: "customer support",
        areaServed: "GB",
        availableLanguage: ["English"],
      },
    ],
  } as const;

  const itemListLd = {
    "@context": "https://schema.org",
    "@type": "ItemList",
    name: `${siteInfo.siteName} Partner Offers`,
    itemListElement: partners.map((partner, index) => ({
      "@type": "ListItem",
      position: index + 1,
      name: partner.name,
      description: partner.welcomeBonus,
      url: `${BASE_URL}/#${partner.name
        .toLowerCase()
        .replace(/[^a-z0-9]+/g, "-")}`,
    })),
  } as const;

  const featuredPartner = partners[0];
  const reviewLd = featuredPartner
    ? {
        "@context": "https://schema.org",
        "@type": "Review",
        itemReviewed: {
          "@type": "Organization",
          name: featuredPartner.name,
          url: `${BASE_URL}/`,
        },
        reviewRating: {
          "@type": "Rating",
          ratingValue: "4.7",
          bestRating: "5",
        },
        author: {
          "@type": "Organization",
          name: siteInfo.siteName,
        },
      }
    : null;

  return (
    <div className="space-y-6 pb-4 sm:space-y-12 sm:pb-6">
      {/* VisibilityDebug only renders in development - never in production */}
      {isDevelopment && (
        <Suspense fallback={null}>
          <VisibilityDebug showAll={showAllPartners} />
        </Suspense>
      )}
      <section className="group relative overflow-hidden rounded-2xl bg-gradient-to-br from-slate-900 via-slate-800 to-slate-900 px-6 py-6 text-white shadow-2xl transition-all duration-700 hover:shadow-[0_20px_50px_rgba(0,0,0,0.5)] sm:rounded-3xl sm:px-12 sm:py-8">
        {/* Animated background gradients with movement */}
        <div className="absolute inset-0 bg-[radial-gradient(circle_at_30%_20%,rgba(120,119,198,0.25),transparent_50%)] transition-all duration-1000 group-hover:opacity-100" />
        <div className="absolute inset-0 bg-[radial-gradient(circle_at_70%_80%,rgba(59,130,246,0.2),transparent_50%)] transition-all duration-1000 group-hover:opacity-100" />
        
        {/* Animated grid pattern */}
        <div className="absolute inset-0 bg-[linear-gradient(rgba(255,255,255,0.03)_1px,transparent_1px),linear-gradient(90deg,rgba(255,255,255,0.03)_1px,transparent_1px)] bg-[size:30px_30px] opacity-60 transition-opacity duration-700 group-hover:opacity-80" />
        
        {/* Shimmer effect on hover */}
        <div className="absolute inset-0 -translate-x-full bg-gradient-to-r from-transparent via-white/20 to-transparent transition-transform duration-1000 group-hover:translate-x-full" />
        
        {/* Glowing border effect */}
        <div className="absolute inset-0 rounded-2xl border border-white/10 transition-all duration-700 group-hover:border-white/20 sm:rounded-3xl" />
        
        <div className="relative space-y-3 text-center sm:space-y-4">
          <span className="inline-flex items-center gap-2 rounded-full border border-white/40 bg-gradient-to-r from-white/20 via-white/15 to-white/20 px-3 py-1.5 text-[10px] font-semibold uppercase tracking-[0.15em] text-white shadow-[0_4px_14px_rgba(255,255,255,0.1)] backdrop-blur-md transition-all duration-300 hover:scale-105 hover:border-white/50 hover:bg-white/25 hover:shadow-[0_6px_20px_rgba(255,255,255,0.15)] sm:gap-2 sm:px-4 sm:py-2 sm:text-xs">
            <span className="relative flex h-1.5 w-1.5 sm:h-2 sm:w-2">
              <span className="absolute inline-flex h-full w-full animate-ping rounded-full bg-emerald-400 opacity-75" />
              <span className="relative inline-flex h-full w-full rounded-full bg-emerald-500 shadow-lg shadow-emerald-500/60" />
            </span>
            Verified Partner Insights
          </span>
          <h1 className="relative text-3xl font-extrabold leading-[1.1] tracking-tight text-white drop-shadow-[0_2px_8px_rgba(0,0,0,0.3)] transition-all duration-500 group-hover:drop-shadow-[0_4px_16px_rgba(255,255,255,0.2)] sm:text-5xl lg:text-6xl">
            <span className="relative inline-block bg-gradient-to-r from-white via-white to-white/95 bg-clip-text text-transparent">
              {siteInfo.tagline}
            </span>
            {/* Subtle underline glow */}
            <span className="absolute -bottom-1 left-1/2 h-0.5 w-0 -translate-x-1/2 bg-gradient-to-r from-transparent via-blue-400 to-transparent transition-all duration-700 group-hover:w-3/4 sm:-bottom-1.5 sm:h-1" />
          </h1>
        </div>
        
        {/* Floating particles effect */}
        <div className="absolute top-1/4 left-1/4 h-1 w-1 rounded-full bg-white/40 opacity-60 animate-pulse" />
        <div className="absolute top-1/3 right-1/3 h-1.5 w-1.5 rounded-full bg-blue-400/40 opacity-50 animate-pulse delay-300" />
        <div className="absolute bottom-1/4 right-1/4 h-1 w-1 rounded-full bg-purple-400/40 opacity-60 animate-pulse delay-700" />
      </section>

      <PartnerShowcase
        partners={partners}
        licensedPartners={licensedPartners}
        unlicensedPartners={unlicensedPartners}
        initialShowAll={showAllPartners}
        showOnlyUnlicensed={showOnlyUnlicensed}
        gclid={gclid}
        accountCode={siteInfo.accountCode}
      />

      <section className="relative space-y-6 overflow-hidden rounded-3xl bg-gradient-to-br from-slate-900 via-slate-800 to-slate-900 px-8 py-10 shadow-2xl sm:px-12 sm:py-12">
        <div className="absolute inset-0 bg-[radial-gradient(circle_at_30%_20%,rgba(120,119,198,0.15),transparent_50%)]" />
        <div className="absolute inset-0 bg-[radial-gradient(circle_at_70%_80%,rgba(59,130,246,0.1),transparent_50%)]" />
        <div className="relative space-y-2 text-center">
          <h2 className="text-3xl font-bold tracking-tight text-white sm:text-4xl">
            Why trust {siteInfo.siteName}?
          </h2>
          <p className="text-base text-white/80 sm:text-lg">
            We&apos;re committed to providing you with the best betting experience
          </p>
        </div>
        <ul className="relative grid gap-4 sm:grid-cols-2 lg:grid-cols-4 sm:gap-6">
          <li className="group relative flex flex-col overflow-hidden rounded-3xl border border-white/20 bg-white/10 backdrop-blur-sm p-6 shadow-lg transition-all duration-500 ease-out hover:-translate-y-2 hover:scale-[1.02] hover:border-white/30 hover:bg-white/15 hover:shadow-xl sm:p-7">
            <div className="relative z-10 mb-3 flex h-14 w-14 items-center justify-center rounded-2xl bg-gradient-to-br from-blue-500 to-blue-600 text-white shadow-lg shadow-blue-500/30 transition-transform duration-500 group-hover:scale-110 mx-auto">
              <svg className="h-7 w-7" fill="none" viewBox="0 0 24 24" stroke="currentColor" strokeWidth={2}>
                <path strokeLinecap="round" strokeLinejoin="round" d="M13 10V3L4 14h7v7l9-11h-7z" />
              </svg>
            </div>
            <h3 className="relative z-10 mb-2 text-center text-xl font-extrabold text-white">Lightning-fast</h3>
            <p className="relative z-10 text-center text-sm leading-relaxed text-white/80">Core Web Vitals optimisation across LCP, CLS, and FID.</p>
          </li>
          <li className="group relative flex flex-col overflow-hidden rounded-3xl border border-white/20 bg-white/10 backdrop-blur-sm p-6 shadow-lg transition-all duration-500 ease-out hover:-translate-y-2 hover:scale-[1.02] hover:border-white/30 hover:bg-white/15 hover:shadow-xl sm:p-7">
            <div className="relative z-10 mb-3 flex h-14 w-14 items-center justify-center rounded-2xl bg-gradient-to-br from-emerald-500 to-emerald-600 text-white shadow-lg shadow-emerald-500/30 transition-transform duration-500 group-hover:scale-110 mx-auto">
              <svg className="h-7 w-7" fill="none" viewBox="0 0 24 24" stroke="currentColor" strokeWidth={2}>
                <path strokeLinecap="round" strokeLinejoin="round" d="M9 12l2 2 4-4m5.618-4.016A11.955 11.955 0 0112 2.944a11.955 11.955 0 01-8.618 3.04A12.02 12.02 0 003 9c0 5.591 3.824 10.29 9 11.622 5.176-1.332 9-6.03 9-11.622 0-1.042-.133-2.052-.382-3.016z" />
              </svg>
            </div>
            <h3 className="relative z-10 mb-2 text-center text-xl font-extrabold text-white">Compliant</h3>
            <p className="relative z-10 text-center text-sm leading-relaxed text-white/80">Manual partner vetting to ensure Google Ads policy compliance.</p>
          </li>
          <li className="group relative flex flex-col overflow-hidden rounded-3xl border border-white/20 bg-white/10 backdrop-blur-sm p-6 shadow-lg transition-all duration-500 ease-out hover:-translate-y-2 hover:scale-[1.02] hover:border-white/30 hover:bg-white/15 hover:shadow-xl sm:p-7">
            <div className="relative z-10 mb-3 flex h-14 w-14 items-center justify-center rounded-2xl bg-gradient-to-br from-purple-500 to-purple-600 text-white shadow-lg shadow-purple-500/30 transition-transform duration-500 group-hover:scale-110 mx-auto">
              <svg className="h-7 w-7" fill="none" viewBox="0 0 24 24" stroke="currentColor" strokeWidth={2}>
                <path strokeLinecap="round" strokeLinejoin="round" d="M12 18h.01M8 21h8a2 2 0 002-2V5a2 2 0 00-2-2H8a2 2 0 00-2 2v14a2 2 0 002 2z" />
              </svg>
            </div>
            <h3 className="relative z-10 mb-2 text-center text-xl font-extrabold text-white">Mobile-first</h3>
            <p className="relative z-10 text-center text-sm leading-relaxed text-white/80">Material-inspired components that convert on any device.</p>
          </li>
          <li className="group relative flex flex-col overflow-hidden rounded-3xl border border-white/20 bg-white/10 backdrop-blur-sm p-6 shadow-lg transition-all duration-500 ease-out hover:-translate-y-2 hover:scale-[1.02] hover:border-white/30 hover:bg-white/15 hover:shadow-xl sm:p-7">
            <div className="relative z-10 mb-3 flex h-14 w-14 items-center justify-center rounded-2xl bg-gradient-to-br from-amber-500 to-amber-600 text-white shadow-lg shadow-amber-500/30 transition-transform duration-500 group-hover:scale-110 mx-auto">
              <svg className="h-7 w-7" fill="none" viewBox="0 0 24 24" stroke="currentColor" strokeWidth={2}>
                <path strokeLinecap="round" strokeLinejoin="round" d="M15 12a3 3 0 11-6 0 3 3 0 016 0z" />
                <path strokeLinecap="round" strokeLinejoin="round" d="M2.458 12C3.732 7.943 7.523 5 12 5c4.478 0 8.268 2.943 9.542 7-1.274 4.057-5.064 7-9.542 7-4.477 0-8.268-2.943-9.542-7z" />
              </svg>
            </div>
            <h3 className="relative z-10 mb-2 text-center text-xl font-extrabold text-white">Transparent</h3>
            <p className="relative z-10 text-center text-sm leading-relaxed text-white/80">Clear disclosure of licensing status for responsible choices.</p>
          </li>
        </ul>
      </section>

      <script
        type="application/ld+json"
        suppressHydrationWarning
        dangerouslySetInnerHTML={{ __html: JSON.stringify(organizationLd) }}
      />
      <script
        type="application/ld+json"
        suppressHydrationWarning
        dangerouslySetInnerHTML={{ __html: JSON.stringify(itemListLd) }}
      />
      {reviewLd ? (
        <script
          type="application/ld+json"
          suppressHydrationWarning
          dangerouslySetInnerHTML={{ __html: JSON.stringify(reviewLd) }}
        />
      ) : null}
    </div>
  );
}
