"use client";

import { useState } from "react";
import Image from "next/image";
import Link from "next/link";
import type { SiteInfo } from "@/types/data";

type HeaderProps = {
  siteInfo: SiteInfo;
};

const navLinks = [
  { href: "/", label: "Home" },
  { href: "/about", label: "About" },
  { href: "/contact", label: "Contact" },
];

export function Header({ siteInfo }: HeaderProps) {
  const [isMenuOpen, setIsMenuOpen] = useState(false);

  return (
    <header className="sticky top-0 z-50 border-b border-slate-300/50 bg-gradient-to-r from-slate-900 via-slate-800 to-slate-900 backdrop-blur-md backdrop-saturate-150 shadow-lg">
      <div className="mx-auto flex max-w-6xl items-center justify-between px-4 py-5 sm:px-6">
        <Link href="/" className="group flex items-center gap-3 transition-opacity hover:opacity-90">
          <span className="relative grid size-14 place-items-center overflow-hidden rounded-2xl border border-white/20 bg-gradient-to-br from-white/10 to-white/5 shadow-lg backdrop-blur-sm transition-transform duration-300 group-hover:scale-105">
            <Image
              src={siteInfo.siteLogo}
              alt={`${siteInfo.siteName} logo`}
              width={56}
              height={56}
              className="h-12 w-12 object-contain"
              priority
              unoptimized={!siteInfo.siteLogo.includes('logo.png')}
            />
          </span>
          <div className="flex flex-col">
            <span className="text-xl font-bold tracking-tight text-white">
              {siteInfo.siteName}
            </span>
            <span className="text-xs font-medium text-white/70">
              {siteInfo.tagline}
            </span>
          </div>
        </Link>
        
        {/* Mobile menu button */}
        <button
          type="button"
          onClick={() => setIsMenuOpen(!isMenuOpen)}
          className="flex flex-col gap-1.5 rounded-xl p-2.5 transition-all hover:bg-white/10 focus-visible:outline focus-visible:outline-2 focus-visible:outline-offset-2 focus-visible:outline-white/50 active:scale-95 sm:hidden"
          aria-label="Toggle menu"
          aria-expanded={isMenuOpen}
        >
          <span
            className={`h-0.5 w-6 origin-center rounded-full bg-white transition-all duration-300 ${
              isMenuOpen ? "translate-y-2 rotate-45" : ""
            }`}
          />
          <span
            className={`h-0.5 w-6 rounded-full bg-white transition-all duration-300 ${
              isMenuOpen ? "opacity-0" : "opacity-100"
            }`}
          />
          <span
            className={`h-0.5 w-6 origin-center rounded-full bg-white transition-all duration-300 ${
              isMenuOpen ? "-translate-y-2 -rotate-45" : ""
            }`}
          />
        </button>

        {/* Desktop navigation */}
        <nav className="hidden items-center gap-2 text-sm font-semibold sm:flex">
          {navLinks.map((link) => (
            <Link
              key={link.href}
              href={link.href}
              className="rounded-xl px-5 py-2.5 text-white/90 transition-all duration-200 hover:bg-white/10 hover:text-white focus-visible:outline focus-visible:outline-2 focus-visible:outline-offset-2 focus-visible:outline-white/50 active:scale-95"
            >
              {link.label}
            </Link>
          ))}
        </nav>
      </div>

      {/* Mobile navigation */}
      {isMenuOpen && (
        <nav className="border-t border-white/20 bg-slate-900/98 backdrop-blur-md sm:hidden">
          <div className="mx-auto flex max-w-6xl flex-col gap-1 px-4 py-3">
            {navLinks.map((link) => (
              <Link
                key={link.href}
                href={link.href}
                onClick={() => setIsMenuOpen(false)}
                className="rounded-xl px-4 py-3.5 text-sm font-semibold text-white transition-all duration-200 hover:bg-white/10 hover:text-white focus-visible:outline focus-visible:outline-2 focus-visible:outline-offset-2 focus-visible:outline-white/50 active:scale-[0.98]"
              >
                {link.label}
              </Link>
            ))}
          </div>
        </nav>
      )}
    </header>
  );
}
