"use client";

import { useMemo } from "react";
import type { Partner } from "@/types/data";
import { PartnerCard } from "./PartnerCard";

type PartnerShowcaseProps = {
  partners: Partner[];
  licensedPartners: Partner[];
  unlicensedPartners: Partner[];
  initialShowAll: boolean;
  showOnlyUnlicensed: boolean;
  gclid: string | null;
  accountCode: string;
};

export function PartnerShowcase({
  partners,
  licensedPartners,
  unlicensedPartners,
  initialShowAll,
  showOnlyUnlicensed,
  gclid,
  accountCode,
}: PartnerShowcaseProps) {
  // Memoize client-side check to avoid re-evaluation
  const hasNavigatorUserAgent = useMemo(
    () => typeof navigator === "undefined" ? true : Boolean(navigator.userAgent),
    []
  );
  const showAll = initialShowAll && hasNavigatorUserAgent;

  // Memoize partner cards with affiliate links
  // gclid is used ONLY for affiliate tracking, never exposed in HTML attributes or metadata
  const partnerCards = useMemo(() => {
    const gclidValue = gclid ?? "organic";
    
    // NEW RULE: If user has gclid (Google Ads), show ONLY unlicensed brands
    let list: Partner[];
    if (showOnlyUnlicensed) {
      list = unlicensedPartners;
    } else if (showAll) {
      list = partners;
    } else {
      list = licensedPartners;
    }
    return list.map((partner) => {
      const linkWithAccount = partner.affiliateLink.replace(
        "{AccountCode}",
        accountCode,
      );
      // gclid is only appended to affiliate links (external), never to internal URLs
      const affiliateHref = `${linkWithAccount}${encodeURIComponent(gclidValue)}`;
      return { partner, affiliateHref };
    });
  }, [accountCode, gclid, licensedPartners, unlicensedPartners, partners, showAll, showOnlyUnlicensed]);

  return (
    <section aria-labelledby="partners-heading" className="space-y-2 sm:space-y-6">
      <div className="grid gap-2.5 sm:gap-6 md:grid-cols-2 lg:gap-8">
        {partnerCards.map(({ partner, affiliateHref }, index) => (
          <PartnerCard
            key={partner.name}
            partner={partner}
            affiliateHref={affiliateHref}
            index={index}
          />
        ))}
      </div>
    </section>
  );
}
