import { readFileSync, existsSync, mkdirSync } from 'fs';
import { join } from 'path';
import type { Partner, SiteInfo } from '@/types/data';

const DATA_DIR = join(process.cwd(), 'data');

/**
 * Ensures the data directory exists
 */
function ensureDataDir() {
  if (!existsSync(DATA_DIR)) {
    mkdirSync(DATA_DIR, { recursive: true });
  }
}

/**
 * Reads partners.json at runtime
 * Filters to only return partners with display: true
 */
export function getPartners(): Partner[] {
  ensureDataDir();
  const filePath = join(DATA_DIR, 'partners.json');
  
  try {
    const fileContent = readFileSync(filePath, 'utf-8');
    const allPartners = JSON.parse(fileContent) as Partner[];
    // Filter to only show partners with display: true (defaults to true if not specified)
    return allPartners.filter((partner) => partner.display !== false);
  } catch (error) {
    console.error('Error reading partners.json:', error);
    return [];
  }
}

/**
 * Reads siteinfo.json at runtime
 */
export function getSiteInfo(): SiteInfo {
  ensureDataDir();
  const filePath = join(DATA_DIR, 'siteinfo.json');
  
  try {
    const fileContent = readFileSync(filePath, 'utf-8');
    return JSON.parse(fileContent) as SiteInfo;
  } catch (error) {
    console.error('Error reading siteinfo.json:', error);
    // Return default values if file doesn't exist
    return {
      siteName: 'BetRanker',
      siteLogo: '/logo.png',
      email: 'support@betranked.com',
      address: 'London, UK',
      phone: '+44 20 1234 5678',
      accountCode: 'BET-UK02KS',
      tagline: 'Top-Rated UK Betting Platforms',
      favicon: '/favicon.ico',
      browserTitle: 'Top-Rated UK Betting Platforms',
    };
  }
}

/**
 * Gets the data directory path
 */
export function getDataDir(): string {
  ensureDataDir();
  return DATA_DIR;
}

