import type { Metadata } from "next";
import { createCanonicalUrl } from "./seo";
import { getSiteInfo } from "./data";

export const BASE_URL = "https://www.betranked.com";

type MetadataArgs = {
  title: string;
  description: string;
  path?: string;
  keywords?: string[];
  /**
   * If true, prevents indexing (for pages with sensitive parameters)
   * Defaults to false - pages are indexable by default
   */
  noindex?: boolean;
};

export const defaultKeywords = [
  "online casino bonuses",
  "best betting sites",
  "uk casino welcome bonus",
  "casino affiliate",
  "betting offers",
];

export function createMetadata({
  title,
  description,
  path = "/",
  keywords = defaultKeywords,
  noindex = false,
}: MetadataArgs): Metadata {
  // Read siteInfo at runtime (not bundled)
  const siteInfo = getSiteInfo();
  
  // Always strip sensitive params from canonical and Open Graph URLs
  const canonicalUrl = createCanonicalUrl(path, BASE_URL);
  const ogUrl = createCanonicalUrl(path, BASE_URL);

  return {
    title,
    description,
    alternates: {
      canonical: canonicalUrl,
    },
    keywords,
    openGraph: {
      title,
      description,
      url: ogUrl, // Clean URL without sensitive params
      siteName: siteInfo.siteName,
      images: [
        {
          url: new URL(siteInfo.siteLogo, BASE_URL).toString(),
          width: 320,
          height: 320,
          alt: `${siteInfo.siteName} logo`,
        },
      ],
      locale: "en_GB",
      type: "website",
    },
    robots: {
      index: !noindex,
      follow: !noindex,
      googleBot: {
        index: !noindex,
        follow: !noindex,
        "max-video-preview": -1,
        "max-image-preview": "large",
        "max-snippet": -1,
      },
    },
  };
}
